// BagGlue.cpp: implementation of the BagGlue class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "BagGlue.h"

extern char gError[100];
extern unsigned int totalbytes;
extern HWND hwnddlg, hwndprogress, hwndedit;
//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

BagGlue::BagGlue(unsigned int size)
{
	if (size > 0)
		ndata = size;
	else
		ndata = BAGDATA_DEFAULTSIZE;

	data = new BAGDATA [ndata];

	header.track = 0;

	FlushCache();
}

BagGlue::~BagGlue()
{
	delete [] data;
}


bool BagGlue::Read(char * filename)
{
	HFILE hfile;
	unsigned int i;

	hfile = (HFILE) CreateFile(filename, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL | FILE_FLAG_SEQUENTIAL_SCAN, NULL);
	if (hfile == HFILE_ERROR)
	{
		wsprintf(gError, "File does not exist.");
		return false;
	}
	if (GetData(hfile, (char *) &header, 16) != 16)
	{
		wsprintf(gError, "File is too small to be valid.");
		CloseHandle((HANDLE) hfile);
		header.track = 0;
		return false;
	}
	if (header.fileID[0] != 'G' || header.fileID[1] != 'A' || 
		header.fileID[2] != 'B' || header.fileID[3] != 'A')
	{
		wsprintf(gError, "File is not in the right format.");
		CloseHandle((HANDLE) hfile);
		header.track = 0;
		return false;
	}
	if (header.track > ndata)
	{
		ndata = header.track;
		delete [] data;
		data = new BAGDATA [ndata];
	}
	i = 0;
	while (i < header.track)
	{
		if (GetData(hfile, data[i].name, 64) != 64)
		{
			wsprintf(gError, "File is incomplete.");
			CloseHandle((HANDLE) hfile);
			header.track = 0;
			return false;
		}
		lstrcpy(data[i].filename, filename);
		i++;
	}
	CloseHandle((HANDLE) hfile);
	return true;
}

bool BagGlue::ReadMP3(char * filename)
{
	HFILE hfile;
	unsigned int i, j;
	char nj[4];
	DWORD hilength;
	char * pstr;

	hfile = (HFILE) CreateFile(filename, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL | FILE_FLAG_SEQUENTIAL_SCAN, NULL);
	if (hfile == HFILE_ERROR)
	{
		wsprintf(gError, "File does not exist.");
		return false;
	}
	FlushCache();
	i = 0;
	memset((void *) nj, 0, 4);
	while (GetCacheData(hfile, nj, 1) == 1)
	{
		i++;
		if ((BYTE) nj[0] != 0xFF)
			continue;
		if (GetCacheData(hfile, &(nj[1]), 1) != 1)
			continue;
		i++;
		if ((((BYTE) nj[1]) & 0xE0) != 0xE0)
			continue;
		if (GetCacheData(hfile, &(nj[2]), 2) != 2)
			continue;
		i += 2;
		if (1 > ndata)
		{
			ndata = 1;
			delete [] data;
			data = new BAGDATA [ndata];
		}
		lstrcpy(data[0].filename, filename);

		j = (((BYTE) nj[1]) & 0x18) >> 3;
		if (j != 1)
		{
			if (j == 0) j = 1;
			if (j == 3) j = 4;
			switch (((BYTE) nj[2]) & 0x0C)
			{
			case 0:
				data[0].freq_khz = 11025 * j;
				break;
			case 4:
				data[0].freq_khz = 12000 * j;
				break;
			case 8:
				data[0].freq_khz = 8000 * j;
				break;
			case 12:
				data[0].freq_khz = 0;
				break;
			}
		}
		data[0].unknown = 0;
		if (j == 2 || j == 1)
		{
			if (((((BYTE) nj[1]) & 0x06) >> 1) != 0)
			{				
				if (((((BYTE) nj[1]) & 0x06) >> 1) < 3)
				{
					if (((((BYTE) nj[2]) & 0xF0) >> 4) == 12)
						data[0].unknown = 0x25;
					else if (((((BYTE) nj[2]) & 0xF0) >> 4) == 8)
						data[0].unknown = 0x24;
				}
				else
				{
					if (((((BYTE) nj[2]) & 0xF0) >> 4) == 8)
						data[0].unknown = 0x25;
					else if (((((BYTE) nj[2]) & 0xF0) >> 4) == 4)
						data[0].unknown = 0x24;
				}
			}
		}
		else if (j == 4)
		{
			if (((((BYTE) nj[2]) & 0xF0) >> 4) == 9 && 
				((((BYTE) nj[1]) & 0x06) >> 1) == 1)
				data[0].unknown = 0x25;
			if (((((BYTE) nj[2]) & 0xF0) >> 4) == 8 && 
				((((BYTE) nj[1]) & 0x06) >> 1) == 2)
				data[0].unknown = 0x25;
			if (((((BYTE) nj[2]) & 0xF0) >> 4) == 4 && 
				((((BYTE) nj[1]) & 0x06) >> 1) == 3)
				data[0].unknown = 0x25;
			if (((((BYTE) nj[2]) & 0xF0) >> 4) == 5 && 
				((((BYTE) nj[1]) & 0x06) >> 1) == 1)
				data[0].unknown = 0x24;
			if (((((BYTE) nj[2]) & 0xF0) >> 4) == 4 && 
				((((BYTE) nj[1]) & 0x06) >> 1) == 2)
				data[0].unknown = 0x24;
			if (((((BYTE) nj[2]) & 0xF0) >> 4) == 2 && 
				((((BYTE) nj[1]) & 0x06) >> 1) == 3)
				data[0].unknown = 0x24;
		}
		if (data[0].unknown == 0)
			data[0].unknown = 0x25;

		data[0].istart[0] = nj[3];
		data[0].istart[1] = nj[2];
		data[0].istart[2] = nj[1];
		data[0].istart[3] = nj[0];
		data[0].location = i - 4;
		hilength = 0;
		data[0].length = GetFileSize((HANDLE) hfile, &hilength);
		if (data[0].length == 0xFFFFFFFF || hilength > 0)
		{
			wsprintf(gError, "File is too big for the bag file.");
			CloseHandle((HANDLE) hfile);
			header.track = 0;
			return false;
		}
		data[0].length -= (i - 4);
		memset((void *) data[0].zero, 0, 12);
		memset((void *) data[0].name, 0, 32);
		CopyDirectory(&pstr, filename);
		i = lstrlen(filename);
		j = lstrlen(pstr);
		if (i - j - 4 >= 32 && *(filename + i - 4) == '.')
			memcpy((void *) data[0].name, filename + j, 31);
		else
			memcpy((char *) data[0].name, filename + j, i - j - 4);
		delete [] pstr;
		memset((void *) header.fileID, 0, 8);
		header.fileID[0] = 0x47;
		header.fileID[1] = 0x41;
		header.fileID[2] = 0x42;
		header.fileID[3] = 0x41;
		header.fileID[4] = 0x04;
		header.track = 1;
		header.unknown = 0x40;
		CloseHandle((HANDLE) hfile);
		return true;
	}

	wsprintf(gError, "File is not in the right format.");
	CloseHandle((HANDLE) hfile);
	header.track = 0;
	return false;
}

bool BagGlue::ReadCMP(char * filename)
{
	HFILE hfile;
	unsigned int i, j;
	char nj[4];
	DWORD hilength;
	char * pstr;

	hfile = (HFILE) CreateFile(filename, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL | FILE_FLAG_SEQUENTIAL_SCAN, NULL);
	if (hfile == HFILE_ERROR)
	{
		wsprintf(gError, "File does not exist.");
		return false;
	}
	FlushCache();
	memset((void *) nj, 0, 4);
	nj[2] = 0x02;
	if (1 > ndata)
	{
		ndata = 1;
		delete [] data;
		data = new BAGDATA [ndata];
	}
	lstrcpy(data[0].filename, filename);

	data[0].freq_khz = 22050;
	data[0].unknown = 0x0C;
	data[0].istart[0] = nj[3];
	data[0].istart[1] = nj[2];
	data[0].istart[2] = nj[1];
	data[0].istart[3] = nj[0];
	data[0].location = 0;
	hilength = 0;
	data[0].length = GetFileSize((HANDLE) hfile, &hilength);
	if (data[0].length == 0xFFFFFFFF || hilength > 0)
	{
		wsprintf(gError, "File is too big for the bag file.");
		CloseHandle((HANDLE) hfile);
		header.track = 0;
		return false;
	}
	memset((void *) data[0].zero, 0, 12);
	memset((void *) data[0].name, 0, 32);
	CopyDirectory(&pstr, filename);
	i = lstrlen(filename);
	j = lstrlen(pstr);
	if (i - j - 4 >= 32 && *(filename + i - 4) == '.')
		memcpy((void *) data[0].name, filename + j, 31);
	else
		memcpy((char *) data[0].name, filename + j, i - j - 4);
	delete [] pstr;
	memset((void *) header.fileID, 0, 8);
	header.fileID[0] = 0x47;
	header.fileID[1] = 0x41;
	header.fileID[2] = 0x42;
	header.fileID[3] = 0x41;
	header.fileID[4] = 0x04;
	header.track = 1;
	header.unknown = 0x40;
	CloseHandle((HANDLE) hfile);
	return true;
}

bool BagGlue::ReadWAV(char * filename)
{
	HFILE hfile;
	unsigned int i, j;
	char riff[] = "RIFF";
	char wave[] = "WAVEfmt ";
	char dat[] = "data";
	RIFFWAVE rw;
	DWORD location;
	DWORD length;
	char gabbage[64];
	char * pstr;

	hfile = (HFILE) CreateFile(filename, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL | FILE_FLAG_SEQUENTIAL_SCAN, NULL);
	if (hfile == HFILE_ERROR)
	{
		wsprintf(gError, "File does not exist.");
		return false;
	}
	FlushCache();
	i = 0;
	if (GetCacheData(hfile, (char *) &rw, 8) != 8)
	{
		wsprintf(gError, "File is too small.");
		CloseHandle((HANDLE) hfile);
		return false;
	}
	if (memcmp(rw.riff, riff, 4))
	{
		wsprintf(gError, "Wave file starts with RIFF.");
		CloseHandle((HANDLE) hfile);
		return false;
	}
	if (GetCacheData(hfile, (char *) rw.wavefmt, 4) != 4)
	{
		wsprintf(gError, "File is too small.");
		CloseHandle((HANDLE) hfile);
		return false;
	}
	if (memcmp(rw.wavefmt, wave, 4))
	{
		wsprintf(gError, "WAVE chunk cannot be found.");
		CloseHandle((HANDLE) hfile);
		return false;
	}
	location = 12;
	while (GetCacheData(hfile, (char *) rw.wavefmt + 4, 4) == 4)
	{
		location += 4;
		if (memcmp(rw.wavefmt + 4, wave + 4, 4))
		{
			if (GetCacheData(hfile, (char *) &length, 4) != 4)
			{
				wsprintf(gError, "File is too small.");
				CloseHandle((HANDLE) hfile);
				return false;
			}
			if (length > 64)
			{
				wsprintf(gError, "There is a wierd chunk in the file.");
				return false;
			}
			if (GetCacheData(hfile, gabbage, length) != length)
			{
				wsprintf(gError, "File is too small.");
				CloseHandle((HANDLE) hfile);
				return false;
			}
			location += (length + 4);
			continue;
		}
		if (GetCacheData(hfile, (char *) &(rw.fmtlength), 4) != 4)
			break;
		if (rw.fmtlength < 16)
			break;
		if (GetCacheData(hfile, (char *) &(rw.pcm), 16) != 16)
			break;
		if (rw.fmtlength > 16)
		{
			if (GetCacheData(hfile, gabbage, rw.fmtlength - 16) != rw.fmtlength - 16)
				break;
		}
		location += (rw.fmtlength + 4);
		while (GetCacheData(hfile, (char *) &(rw.data), 4) == 4)
		{
			location += 4;
			if (memcmp(rw.data, dat, 4))
			{
				if (GetCacheData(hfile, (char *) &length, 4) != 4)
				{
					wsprintf(gError, "File is too small.");
					CloseHandle((HANDLE) hfile);
					return false;
				}
				if (length > 64)
				{
					wsprintf(gError, "There is a wierd chunk in the file.");
					return false;
				}
				if (GetCacheData(hfile, gabbage, length) != length)
				{
					wsprintf(gError, "File is too small.");
					CloseHandle((HANDLE) hfile);
					return false;
				}
				location += (length + 4);
				continue;
			}
			if (GetCacheData(hfile, (char *) &(rw.datalength), 4) != 4)
				break;
			if (rw.pcm != 0x0001)
			{
				wsprintf(gError, "Only RAW PCM WAVE files are supported.");
				CloseHandle((HANDLE) hfile);
				return false;
			}
			if (1 > ndata)
			{
				ndata = 1;
				delete [] data;
				data = new BAGDATA [ndata];
			}
			memset((void *) data[0].name, 0, 32);
			CopyDirectory(&pstr, filename);
			i = lstrlen(filename);
			j = lstrlen(pstr);
			if (i - j - 4 >= 32 && *(filename + i - 4) == '.')
				memcpy((void *) data[0].name, filename + j, 31);
			else
				memcpy((char *) data[0].name, filename + j, i - j - 4);
			delete [] pstr;
			data[0].location = location;
			data[0].length = rw.datalength;
			data[0].freq_khz = rw.rate;
			data[0].unknown = 6;
			memset((void *) data[0].istart, 0, 4);
			memset((void *) data[0].zero, 0, 12);
			lstrcpy(data[0].filename, filename);
			memset((void *) header.fileID, 0, 8);
			header.fileID[0] = 0x47;
			header.fileID[1] = 0x41;
			header.fileID[2] = 0x42;
			header.fileID[3] = 0x41;
			header.fileID[4] = 0x04;
			header.track = 1;
			header.unknown = 0x40;
			CloseHandle((HANDLE) hfile);
			return true;
		}
		break;
	}
	wsprintf(gError, "fmt or data chunk cannot be found.");
	CloseHandle((HANDLE) hfile);
	return false;
}

bool BagGlue::Write(char * filename)
{
	bool tempcreate = false;
	char * tempfilename;
	char * tempfile = "delete.me";
	BAGDATA bagdata;
	HFILE hfile1, hfile2;
	DWORD nbyte, nbyte2, nbyte3, nstart, nlength;
	char buffer[512], str[50];
	unsigned int i;

	hfile2 = (HFILE) CreateFile(filename, GENERIC_WRITE, 0, NULL, CREATE_NEW, FILE_ATTRIBUTE_NORMAL | FILE_FLAG_WRITE_THROUGH, NULL);
	if (((HANDLE) hfile2) == INVALID_HANDLE_VALUE)
	{
		if (!CopyDirectory(&tempfilename, filename))
		{
			wsprintf(gError, "Program runs out of memory.");
			return false;
		}
		lstrcat(tempfilename, tempfile);
		hfile2 = (HFILE) CreateFile(tempfilename, GENERIC_WRITE, 0, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL | FILE_FLAG_WRITE_THROUGH, NULL);
		tempcreate = true;
	}
	WriteFile((HANDLE) hfile2, (LPVOID) &header, 16, &nbyte, NULL);
	if (nbyte != 16)
	{
		wsprintf(gError, "Cannot write output file.");
		CloseHandle((HANDLE) hfile2);
		if (tempcreate) delete [] tempfilename;
		return false;
	}
	nstart = header.track * 64 + 16;
	i = 0;
	while (i < header.track)
	{
		memcpy((void *) &bagdata, &(data[i]), 64);
		bagdata.location = nstart;
		nstart += data[i].length;
		WriteFile((HANDLE) hfile2, (LPCVOID) &bagdata, 64, &nbyte, NULL);
		if (nbyte != 64)
		{
			wsprintf(gError, "Cannot write output file.");
			CloseHandle((HANDLE) hfile2);
			if (tempcreate) delete [] tempfilename;
			return false;
		}
		i++;
	}
	i = 0;
	nbyte3 = 0;
	hfile1 = (HFILE) CreateFile(data[i].filename, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL | FILE_FLAG_SEQUENTIAL_SCAN, NULL);
	if (hfile1 == HFILE_ERROR)
	{
		wsprintf(gError, "Input file does not exist.");
		CloseHandle((HANDLE) hfile2);
		if (tempcreate) delete [] tempfilename;
		return false;
	}
	while (1)
	{
		if (SetFilePointer((HANDLE) hfile1, data[i].location, NULL, FILE_BEGIN) == 0xffffffff)
		{
				wsprintf(gError, "Cannot read input file.");
				CloseHandle((HANDLE) hfile2);
				CloseHandle((HANDLE) hfile1);
				if (tempcreate) delete [] tempfilename;
				return false;
		}
		nbyte3 += nlength = data[i].length;
		while (nlength > 512)
		{
			nbyte = GetData(hfile1, buffer, 512);
			WriteFile((HANDLE) hfile2, (LPVOID) buffer, 512, &nbyte, NULL);
			if (nbyte != 512)
			{
				wsprintf(gError, "Cannot write output file.");
				CloseHandle((HANDLE) hfile2);
				CloseHandle((HANDLE) hfile1);
				if (tempcreate) delete [] tempfilename;
				return false;
			}
			nlength -= 512;
		}
		nbyte = GetData(hfile1, buffer, nlength);
		WriteFile((HANDLE) hfile2, (LPVOID) buffer, nbyte, &nbyte2, NULL);
		if (nbyte != nbyte2)
		{
			wsprintf(gError, "Cannot write output file.");
			CloseHandle((HANDLE) hfile2);
			CloseHandle((HANDLE) hfile1);
			if (tempcreate) delete [] tempfilename;
			return false;
		}

		if (hwnddlg != NULL)
		{
			SendMessage(hwndprogress, PBM_STEPIT, (WPARAM) 0, (LPARAM) 0);
			wsprintf(str, "%u / %u", nbyte3, totalbytes);
			SendMessage(hwndedit, WM_SETTEXT, (WPARAM) 0, (LPARAM) str);
		}

		i++;
		if (i == header.track)
			break;
		if (lstrcmp(data[i-1].filename, data[i].filename) != 0)
		{
			CloseHandle((HANDLE) hfile1);
			hfile1 = (HFILE) CreateFile(data[i].filename, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL | FILE_FLAG_SEQUENTIAL_SCAN, NULL);
			if (hfile1 == HFILE_ERROR)
			{
				wsprintf(gError, "Input file does not exist.");
				CloseHandle((HANDLE) hfile2);
				if (tempcreate) delete [] tempfilename;
				return false;
			}
		}
	}
	CloseHandle((HANDLE) hfile2);
	CloseHandle((HANDLE) hfile1);

	if (tempcreate == true)
	{
		DeleteFile(filename);
		MoveFile(tempfilename, filename);
		delete [] tempfilename;
	}
	return true;
}

bool BagGlue::WriteMemory(void * hMem, unsigned int index)
{
	HFILE hfile1;
	DWORD nbyte, nbyte2, nlength;

	if (index > header.track)
	{
		wsprintf(gError, "Track is out of range.");
		return false;
	}
	hfile1 = (HFILE) CreateFile(data[index].filename, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL | FILE_FLAG_SEQUENTIAL_SCAN, NULL);
	if (hfile1 == HFILE_ERROR)
	{
		wsprintf(gError, "Input file does not exist.");
		return false;
	}
	if (SetFilePointer((HANDLE) hfile1, data[index].location, NULL, FILE_BEGIN) == 0xffffffff)
	{
			wsprintf(gError, "Cannot read input file.");
			CloseHandle((HANDLE) hfile1);
			return false;
	}
	nlength = data[index].length;
	nbyte2 = 0;
	while (nlength > 512)
	{
		nbyte = GetData(hfile1, (char *) hMem + nbyte2, 512);
		nlength -= nbyte;
		nbyte2 += nbyte;
	}
	nbyte = GetData(hfile1, (char *) hMem + nbyte2, nlength);
	CloseHandle((HANDLE) hfile1);
	return true;
}

bool BagGlue::Write(char * filename, unsigned int index)
{
	bool tempcreate = false;
	char * tempfilename;
	char * tempfile = "delete.me";
	HFILE hfile1, hfile2;
	DWORD nbyte, nbyte2, nlength;
	char buffer[512];

	if (index > header.track)
	{
		wsprintf(gError, "Track is out of range.");
		return false;
	}
	hfile2 = (HFILE) CreateFile(filename, GENERIC_WRITE, 0, NULL, CREATE_NEW, FILE_ATTRIBUTE_NORMAL | FILE_FLAG_WRITE_THROUGH, NULL);
	if (((HANDLE) hfile2) == INVALID_HANDLE_VALUE)
	{
		if (!CopyDirectory(&tempfilename, filename))
		{
			wsprintf(gError, "Program runs out of memory.");
			return false;
		}
		lstrcat(tempfilename, tempfile);
		hfile2 = (HFILE) CreateFile(tempfilename, GENERIC_WRITE, 0, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL | FILE_FLAG_WRITE_THROUGH, NULL);
		tempcreate = true;
	}
	hfile1 = (HFILE) CreateFile(data[index].filename, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL | FILE_FLAG_SEQUENTIAL_SCAN, NULL);
	if (hfile1 == HFILE_ERROR)
	{
		wsprintf(gError, "Input file does not exist.");
		CloseHandle((HANDLE) hfile2);
		if (tempcreate) delete [] tempfilename;
		return false;
	}
	if (SetFilePointer((HANDLE) hfile1, data[index].location, NULL, FILE_BEGIN) == 0xffffffff)
	{
			wsprintf(gError, "Cannot read input file.");
			CloseHandle((HANDLE) hfile2);
			CloseHandle((HANDLE) hfile1);
			if (tempcreate) delete [] tempfilename;
			return false;
	}
	nlength = data[index].length;
	while (nlength > 512)
	{
		nbyte = GetData(hfile1, buffer, 512);
		WriteFile((HANDLE) hfile2, (LPVOID) buffer, 512, &nbyte, NULL);
		if (nbyte != 512)
		{
			wsprintf(gError, "Cannot write output file.");
			CloseHandle((HANDLE) hfile2);
			CloseHandle((HANDLE) hfile1);
			if (tempcreate) delete [] tempfilename;
			return false;
		}
		nlength -= 512;
	}
	nbyte = GetData(hfile1, buffer, nlength);
	WriteFile((HANDLE) hfile2, (LPVOID) buffer, nbyte, &nbyte2, NULL);
	if (nbyte != nbyte2)
	{
		wsprintf(gError, "Cannot write output file.");
		CloseHandle((HANDLE) hfile2);
		CloseHandle((HANDLE) hfile1);
		if (tempcreate) delete [] tempfilename;
		return false;
	}
	CloseHandle((HANDLE) hfile2);
	CloseHandle((HANDLE) hfile1);

	if (tempcreate == true)
	{
		DeleteFile(filename);
		MoveFile(tempfilename, filename);
		delete [] tempfilename;
	}
	return true;
}

bool BagGlue::WriteWAVMemory(void * hMem, unsigned int index)
{
	HFILE hfile1;
	DWORD nbyte, nbyte2, nlength;
	RIFFWAVE riff;

	if (index > header.track)
	{
		wsprintf(gError, "Track is out of range.");
		return false;
	}
	riff.riff[0] = 'R';
	riff.riff[1] = 'I';
	riff.riff[2] = 'F';
	riff.riff[3] = 'F';
	riff.length = data[index].length + 36;
	riff.wavefmt[0] = 'W';
	riff.wavefmt[1] = 'A';
	riff.wavefmt[2] = 'V';
	riff.wavefmt[3] = 'E';
	riff.wavefmt[4] = 'f';
	riff.wavefmt[5] = 'm';
	riff.wavefmt[6] = 't';
	riff.wavefmt[7] = ' ';
	riff.fmtlength = 0x00000010;
	riff.data[0] = 'd';
	riff.data[1] = 'a';
	riff.data[2] = 't';
	riff.data[3] = 'a';

	if ((BYTE)(data[index].unknown) == 0x06 || (BYTE)(data[index].unknown) == 0x16)
	{
		riff.pcm = 0x0001;
		riff.stereo = 0x0001;
		riff.rate = data[index].freq_khz;
		riff.data3 = 16;
		riff.data2 = riff.stereo * riff.data3 / 8;
		riff.data1 = riff.rate * riff.data2;
	}
	else if ((BYTE)(data[index].unknown) == 0x02)
	{
		riff.pcm = 0x0001;
		riff.stereo = 0x0001;
		riff.rate = data[index].freq_khz;
		riff.data3 = 8;
		riff.data2 = riff.stereo * riff.data3 / 8;
		riff.data1 = riff.rate * riff.data2;
	}
	else if ((BYTE)(data[index].unknown) == 0x0C || (BYTE)(data[index].unknown) == 0x16)
	{
		riff.pcm = 0x0001;
		riff.stereo = 0x0001;
		riff.rate = data[index].freq_khz / 2;
		riff.data3 = 8;
		riff.data2 = riff.stereo * riff.data3 / 8;
		riff.data1 = riff.rate * riff.data2;
	}
	else
	{
		riff.pcm = 0xffff;
		riff.stereo = 0x0002;
		riff.rate = data[index].freq_khz;
		riff.data3 = 0;
		riff.data2 = riff.stereo * riff.data3 / 8;
		riff.data1 = riff.rate * riff.data2;
	}
	riff.datalength = data[index].length;

	memcpy(hMem, (LPVOID) &riff, 44);
	
	hfile1 = (HFILE) CreateFile(data[index].filename, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL | FILE_FLAG_SEQUENTIAL_SCAN, NULL);
	if (hfile1 == HFILE_ERROR)
	{
		wsprintf(gError, "Input file does not exist.");
		return false;
	}
	if (SetFilePointer((HANDLE) hfile1, data[index].location, NULL, FILE_BEGIN) == 0xffffffff)
	{
			wsprintf(gError, "Cannot read input file.");
			CloseHandle((HANDLE) hfile1);
			return false;
	}
	nlength = data[index].length;
	nbyte2 = 44;
	while (nlength > 512)
	{
		nbyte = GetData(hfile1, (char *) hMem + nbyte2, 512);
		nlength -= nbyte;
		nbyte2 += nbyte;
	}
	nbyte = GetData(hfile1, (char *) hMem + nbyte2, nlength);
	CloseHandle((HANDLE) hfile1);
	return true;
}

bool BagGlue::WriteWAV(char * filename, unsigned int index)
{
	bool tempcreate = false;
	char * tempfilename;
	char * tempfile = "delete.me";
	HFILE hfile1, hfile2;
	DWORD nbyte, nbyte2, nlength;
	char buffer[512];
	RIFFWAVE riff;

	if (index > header.track)
	{
		wsprintf(gError, "Track is out of range.");
		return false;
	}
	hfile2 = (HFILE) CreateFile(filename, GENERIC_WRITE, 0, NULL, CREATE_NEW, FILE_ATTRIBUTE_NORMAL | FILE_FLAG_WRITE_THROUGH, NULL);
	if (((HANDLE) hfile2) == INVALID_HANDLE_VALUE)
	{
		if (!CopyDirectory(&tempfilename, filename))
		{
			wsprintf(gError, "Program runs out of memory.");
			return false;
		}
		lstrcat(tempfilename, tempfile);
		hfile2 = (HFILE) CreateFile(tempfilename, GENERIC_WRITE, 0, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL | FILE_FLAG_WRITE_THROUGH, NULL);
		tempcreate = true;
	}
	riff.riff[0] = 'R';
	riff.riff[1] = 'I';
	riff.riff[2] = 'F';
	riff.riff[3] = 'F';
	riff.length = data[index].length + 36;
	riff.wavefmt[0] = 'W';
	riff.wavefmt[1] = 'A';
	riff.wavefmt[2] = 'V';
	riff.wavefmt[3] = 'E';
	riff.wavefmt[4] = 'f';
	riff.wavefmt[5] = 'm';
	riff.wavefmt[6] = 't';
	riff.wavefmt[7] = ' ';
	riff.fmtlength = 0x00000010;
	riff.data[0] = 'd';
	riff.data[1] = 'a';
	riff.data[2] = 't';
	riff.data[3] = 'a';

	if ((BYTE)(data[index].unknown) == 0x06 || (BYTE)(data[index].unknown) == 0x16)
	{
		riff.pcm = 0x0001;
		riff.stereo = 0x0001;
		riff.rate = data[index].freq_khz;
		riff.data3 = 16;
		riff.data2 = riff.stereo * riff.data3 / 8;
		riff.data1 = riff.rate * riff.data2;
	}
	else if ((BYTE)(data[index].unknown) == 0x02)
	{
		riff.pcm = 0x0001;
		riff.stereo = 0x0001;
		riff.rate = data[index].freq_khz;
		riff.data3 = 8;
		riff.data2 = riff.stereo * riff.data3 / 8;
		riff.data1 = riff.rate * riff.data2;
	}
	else if ((BYTE)(data[index].unknown) == 0x0C || (BYTE)(data[index].unknown) == 0x16)
	{
		riff.pcm = 0x0001;
		riff.stereo = 0x0001;
		riff.rate = data[index].freq_khz / 2;
		riff.data3 = 8;
		riff.data2 = riff.stereo * riff.data3 / 8;
		riff.data1 = riff.rate * riff.data2;
	}
	else
	{
		riff.pcm = 0xffff;
		riff.stereo = 0x0002;
		riff.rate = data[index].freq_khz;
		riff.data3 = 0;
		riff.data2 = riff.stereo * riff.data3 / 8;
		riff.data1 = riff.rate * riff.data2;
	}
	riff.datalength = data[index].length;

	WriteFile((HANDLE) hfile2, (LPVOID) &riff, 44, &nbyte, NULL);
	
	hfile1 = (HFILE) CreateFile(data[index].filename, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL | FILE_FLAG_SEQUENTIAL_SCAN, NULL);
	if (hfile1 == HFILE_ERROR)
	{
		wsprintf(gError, "Input file does not exist.");
		CloseHandle((HANDLE) hfile2);
		if (tempcreate) delete [] tempfilename;
		return false;
	}
	if (SetFilePointer((HANDLE) hfile1, data[index].location, NULL, FILE_BEGIN) == 0xffffffff)
	{
			wsprintf(gError, "Cannot read input file.");
			CloseHandle((HANDLE) hfile2);
			CloseHandle((HANDLE) hfile1);
			if (tempcreate) delete [] tempfilename;
			return false;
	}
	nlength = data[index].length;
	while (nlength > 512)
	{
		nbyte = GetData(hfile1, buffer, 512);
		WriteFile((HANDLE) hfile2, (LPVOID) buffer, 512, &nbyte, NULL);
		if (nbyte != 512)
		{
			wsprintf(gError, "Cannot write output file.");
			CloseHandle((HANDLE) hfile2);
			CloseHandle((HANDLE) hfile1);
			if (tempcreate) delete [] tempfilename;
			return false;
		}
		nlength -= 512;
	}
	nbyte = GetData(hfile1, buffer, nlength);
	WriteFile((HANDLE) hfile2, (LPVOID) buffer, nbyte, &nbyte2, NULL);
	if (nbyte != nbyte2)
	{
		wsprintf(gError, "Cannot write output file.");
		CloseHandle((HANDLE) hfile2);
		CloseHandle((HANDLE) hfile1);
		if (tempcreate) delete [] tempfilename;
		return false;
	}
	CloseHandle((HANDLE) hfile2);
	CloseHandle((HANDLE) hfile1);

	if (tempcreate == true)
	{
		DeleteFile(filename);
		MoveFile(tempfilename, filename);
		delete [] tempfilename;
	}
	return true;
}

bool BagGlue::Close()
{
	header.track = 0;
	return true;
}

bool BagGlue::Combine(BagGlue & bg)
{
	BAGDATA * temp;
	unsigned int i, j;
	if (bg.header.track == 0)
	{
		lstrcpy(gError, "There is no track to combine.");
		return false;
	}
	if (header.track == 0)
	{
		memcpy((void *) &header, &(bg.header), sizeof(BAGHEADER));
		header.track = 0;
	}
	if (header.track + bg.header.track > ndata)
	{
		ndata = header.track + bg.header.track;
		temp = new BAGDATA [ndata];
		for (i = 0; i < header.track; i++)
		{
			memcpy((void *) &(temp[i]), &(data[i]), sizeof(BAGDATA));
		}
		delete [] data;
		data = temp;
	}
	i = header.track;
	j = 0;
	header.track += bg.header.track;
	while (i < header.track)
	{
		memcpy((void *) &(data[i]), &(bg.data[j]), sizeof(BAGDATA));
		i++;
		j++;
	}
	return true;
}

bool BagGlue::Erase(unsigned int index)
{
	unsigned int i;
	if (index >= header.track)
	{
		lstrcpy(gError, "Index out of range.");
		return false;
	}
	header.track--;
	for (i = index; i < header.track; i++)
	{
		memcpy((void *) &(data[i]), &(data[i+1]), sizeof(BAGDATA));
	}
	return true;
}

bool BagGlue::Erase(unsigned int * tracks, unsigned int count)
{
	unsigned int i;
	if (count == header.track)
	{
		header.track = 0;
		return true;
	}
	i = 0;
	while (i < count)
	{
		if (!Erase(tracks[i]))
			return false;
		i++;
	}
	return true;
}

DWORD BagGlue::GetData(HFILE hfile, char * buffer, DWORD length)
{
	DWORD nBytesRead;

	if (!ReadFile((HANDLE) hfile, buffer, length, &nBytesRead, NULL))
		return 0;
	
	return nBytesRead;
}

DWORD BagGlue::GetCacheData(HFILE hfile, char * buffer, DWORD length)
{
	DWORD nBytesRead;
	if (cacheBytesRead == 0)
	{
		cacheBytesRead = GetData(hfile, cache, 64);
		cacheindex = 0;
		if (cacheBytesRead == 0) return 0;
	}
	if (length > cacheBytesRead)
	{
		memcpy((void *) buffer, cache + cacheindex, cacheBytesRead);
		nBytesRead = GetData(hfile, (char *) (buffer + cacheBytesRead), 
			length - cacheBytesRead);
		length = nBytesRead + cacheBytesRead;
		cacheindex = 64;
		cacheBytesRead = 0;
		return length;
	}
	memcpy((void *) buffer, cache + cacheindex, length);
	cacheBytesRead -= length;
	cacheindex += length;
	return length;
}

void BagGlue::FlushCache()
{
	cacheindex = 512;
	cacheBytesRead = 0;
}

unsigned int BagGlue::GetTotalTracks()
{
	return header.track;
}

char * BagGlue::GetTrackName(unsigned int index)
{
	return data[index].name;
}

unsigned int BagGlue::GetTrackFreq(unsigned int index)
{
	return data[index].freq_khz;
}

unsigned int BagGlue::GetTrackLen(unsigned int index)
{
	return data[index].length;
}

char * BagGlue::GetTrackFile(unsigned int index)
{
	return data[index].filename;
}

bool BagGlue::CopyDirectory(char ** tempfilename, char * filename)
{
	unsigned int i;
	i = lstrlen(filename);
	*tempfilename = new char [i+10];
	if (tempfilename == NULL)
		return false;
	i--;
	while (i > 0)
	{
		if (filename[i] == '\\')
		{
			lstrcpyn(*tempfilename, filename, i + 2);
			return true;
		}
		i--;
	}
	(*tempfilename)[0] = 0;
	return true;
}

bool BagGlue::IsMP3(unsigned int index)
{
	if ((BYTE)(data[index].istart[3]) != 0xFF)
		return false;
	return true;
}

bool BagGlue::IsWAV(unsigned int index)
{
	if ((BYTE)(data[index].unknown) != 0x06)
		if ((BYTE)(data[index].unknown) != 0x0C)
			if ((BYTE)(data[index].unknown) != 0x1C)
				if ((BYTE)(data[index].unknown) != 0x02)
					if ((BYTE)(data[index].unknown) != 0x16)
						return false;
	return true;
}